<!-- ===========================
   Custom Package Builder Page
=========================== -->
<main class="flex-grow">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-10">
        <!-- Page Header -->
        <div class="mb-8">
            <h1 class="text-4xl font-bold text-slate-800">Create Custom Package</h1>
            <p class="text-slate-500 mt-2">Build a personalized itinerary for your clients.</p>
        </div>

        <div class="flex flex-col lg:flex-row gap-8">
            <!-- LEFT COLUMN -->
            <div class="flex-grow lg:w-2/3 bg-white p-6 rounded-lg shadow-md border border-slate-200">
                <form id="custom-package-form">

                    <!-- Basic Info -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 border-b border-slate-200 pb-6">
                        <div>
                            <label for="package-name" class="block text-sm font-medium text-slate-700">Package Name</label>
                            <input type="text" id="package-name" placeholder="e.g., Summer Goa Trip"
                                class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                        </div>
                        <div>
                            <label for="start-date" class="block text-sm font-medium text-slate-700">Start Date</label>
                            <input type="date" id="start-date"
                                class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                        </div>
                        <div>
                            <label for="destination" class="block text-sm font-medium text-slate-700">Destination</label>
                            <select id="destination"
                                class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                                <option value="0">Select Destination</option>
                                <?php foreach($destinations as $d): ?>
                                    <option value="<?= $d['id'] ?>"><?= $d['name'] ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <!-- Guest Info -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 border-b border-slate-200 py-6">
                        <div>
                            <label for="adults" class="block text-sm font-medium text-slate-700">Adults</label>
                            <input type="number" id="adults" value="2" min="1"
                                class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                        </div>
                        <div>
                            <label for="children" class="block text-sm font-medium text-slate-700">Children</label>
                            <input type="number" id="children" value="0" min="0"
                                class="mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                        </div>
                    </div>

                    <!-- Itinerary Rows -->
                    <div class="pt-6">
                        <h3 class="text-lg font-semibold text-slate-800 mb-4">Itinerary Builder</h3>
                        <div id="itinerary-rows" class="space-y-4"></div>

                        <button type="button" id="add-row-btn"
                            class="mt-4 flex items-center space-x-2 text-sm font-medium text-sky-600 hover:text-sky-700">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"
                                xmlns="http://www.w3.org/2000/svg">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                                    d="M12 9v3m0 0v3m0-3h3m-3 0H9m12 0a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <span>Add City / Leg</span>
                        </button>
                    </div>
                </form>
            </div>

            <!-- RIGHT COLUMN -->
            <div class="lg:w-1/3">
                <div class="sticky top-24">
                    <div class="bg-white rounded-lg shadow-md border border-slate-200">
                        <div class="p-6 border-b border-slate-200">
                            <h3 class="text-lg font-bold text-slate-800">Booking Summary</h3>
                            <p id="summary-package-name" class="text-slate-500 text-sm mt-1">Please name your package</p>
                        </div>
                        <div class="p-6 space-y-4">
                            <div id="summary-itinerary">
                                <h4 class="font-semibold text-slate-700 mb-2">Itinerary</h4>
                                <p class="text-sm text-slate-500">Add items to see summary...</p>
                            </div>

                            <div class="border-t border-slate-200 pt-4 space-y-2">
                                <div class="flex justify-between text-sm">
                                    <span class="text-slate-500">Subtotal</span>
                                    <span id="summary-subtotal" class="font-medium text-slate-700">₹0</span>
                                </div>
                                <div class="flex justify-between text-lg font-bold">
                                    <span class="text-slate-800">Total Price</span>
                                    <span id="summary-total" class="text-sky-600">₹0</span>
                                </div>
                            </div>

                            <button id="submit-inquiry"
                                class="w-full bg-orange-500 text-white font-bold py-3 px-4 rounded-md hover:bg-orange-600 transition-colors">
                                Submit Inquiry
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</main>

<!-- ================================
  Itinerary Template (City > Hotel > Room)
================================ -->
<template id="itinerary-row-template">
    <div class="itinerary-row grid grid-cols-1 md:grid-cols-4 gap-4 p-4 border rounded-md bg-slate-50 items-end">
        <div>
            <label class="block text-sm font-medium text-slate-700">City</label>
            <select class="city-select mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                <option value="0">Select City</option>
            </select>
        </div>
        <div>
            <label class="block text-sm font-medium text-slate-700">Hotel</label>
            <select class="hotel-select mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                <option value="0" data-price="0">Select Hotel</option>
            </select>
        </div>
        <div>
            <label class="block text-sm font-medium text-slate-700">Room</label>
            <select class="room-select mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
                <option value="0" data-price="0">Select Room</option>
            </select>
        </div>
        <div class="flex items-center gap-2">
            <div class="flex-grow">
                <label class="block text-sm font-medium text-slate-700">Nights</label>
                <input type="number" value="1" min="1"
                    class="nights-input mt-1 block w-full rounded-md border-slate-300 shadow-sm focus:border-sky-500 focus:ring-sky-500 sm:text-sm">
            </div>
            <button type="button"
                class="remove-row-btn text-red-500 hover:text-red-700 p-2 rounded-md bg-red-100 hover:bg-red-200">
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
                        d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                </svg>
            </button>
        </div>
    </div>
</template>

<!-- =============================
         SCRIPT SECTION
============================= -->
<script>
$(document).ready(function(){
    const baseUrl = '<?= base_url() ?>';

    // -------------------------------
    // LOAD CITIES FUNCTION
    // -------------------------------
    function loadCitiesForDestination(destinationId, targetRow = null) {
        if (!destinationId || destinationId == 0) return;

        $.ajax({
            url: baseUrl + 'website/get_cities_by_destination',
            type: 'POST',
            data: { destination_id: destinationId },
            dataType: 'json',
            success: function(cities){
                const populate = (selectEl) => {
                    selectEl.html('<option value="0">Select City</option>');
                    $.each(cities, function(i, city){
                        selectEl.append(`<option value="${city.id}">${city.name}</option>`);
                    });
                };

                if (targetRow) populate(targetRow.find('.city-select'));
                else $('.city-select').each(function(){ populate($(this)); });
            },
            error: function(){
                alert('Error loading cities');
            }
        });
    }

    // -------------------------------
    // ADD ROW FUNCTION
    // -------------------------------
    function addItineraryRow() {
        const row = $($('#itinerary-row-template').html());
        $('#itinerary-rows').append(row);
        const destinationId = $('#destination').val();
        if (destinationId && destinationId != 0) loadCitiesForDestination(destinationId, row);
    }

    addItineraryRow(); // default row

    $(document).on('click', '.remove-row-btn', function(){
        $(this).closest('.itinerary-row').remove();
        updateSummary();
    });

    $('#destination').on('change', function(){
        const destinationId = $(this).val();
        loadCitiesForDestination(destinationId);
    });

    // -------------------------------
    // FETCH HOTELS & ROOMS
    // -------------------------------
    $(document).on('change', '.city-select', function(){
        const cityId = $(this).val();
        const hotelSelect = $(this).closest('.itinerary-row').find('.hotel-select');
        const roomSelect = $(this).closest('.itinerary-row').find('.room-select');

        if (cityId == 0) {
            hotelSelect.html('<option value="0">Select Hotel</option>');
            roomSelect.html('<option value="0">Select Room</option>');
            return;
        }

        $.ajax({
            url: baseUrl + 'website/get_hotels_by_city',
            type: 'POST',
            data: { city_id: cityId },
            dataType: 'json',
            success: function(hotels){
                hotelSelect.html('<option value="0" data-price="0">Select Hotel</option>');
                $.each(hotels, function(i, h){
                    hotelSelect.append(`<option value="${h.id}" data-price="0">${h.name}</option>`);
                });
            }
        });
    });

    $(document).on('change', '.hotel-select', function(){
        const hotelId = $(this).val();
        const roomSelect = $(this).closest('.itinerary-row').find('.room-select');
        if (hotelId == 0) { roomSelect.html('<option value="0">Select Room</option>'); return; }

        $.ajax({
            url: baseUrl + 'website/get_rooms_by_hotel',
            type: 'POST',
            data: { property_id: hotelId },
            dataType: 'json',
            success: function(rooms){
                roomSelect.html('<option value="0">Select Room</option>');
                $.each(rooms, function(i, room){
                    roomSelect.append(`<option value="${room.id}" data-price="0">${room.name}</option>`);
                });
            }
        });
    });

    $(document).on('change', '.room-select', function(){
        const row = $(this).closest('.itinerary-row');
        const roomId = $(this).val();
        const startDate = $('#start-date').val();

        // validation
        if (!startDate) {
            alert('Please select start date first');
            $('#start-date').focus();
            $(this).val(0);
            return;
        }

        // const today = new Date().toISOString().split('T')[0];
        // if (startDate < today) {
        //     alert('Start date cannot be in the past');
        //     $('#start-date').focus();
        //     return;
        // }

        // if (!roomId || roomId == 0) return;
        if (!roomId) return;

        $.ajax({
            url: baseUrl + 'website/get_room_price',
            type: 'POST',
            data: { room_id: roomId, start_date: startDate },
            dataType: 'json',
            success: function(res){
                const price = parseFloat(res.price_with_markup || 0);
                row.find('.room-select option:selected').attr('data-price', price);
                updateSummary();
            }
        });
    });


    $(document).on('change keyup', '.nights-input, #start-date', function(){
        updateSummary();
    });

    $('#add-row-btn').on('click', function(){
        addItineraryRow();
    });

    // -------------------------------
    // SUMMARY CALCULATION
    // -------------------------------
    // function updateSummary(){
    //     let subtotal = 0, nightsTotal = 0;
    //     let itineraryHtml = '';

    //     $('#itinerary-rows .itinerary-row').each(function(){
    //         const hotel = $(this).find('.hotel-select option:selected').text();
    //         const room = $(this).find('.room-select option:selected').text();
    //         const nights = parseInt($(this).find('.nights-input').val() || 1);
    //         const price = parseFloat($(this).find('.room-select option:selected').attr('data-price') || 0);
    //         const total = price * nights;

    //         if (price > 0) {
    //             itineraryHtml += `<li class="flex justify-between text-sm">
    //                     <span>${hotel} (${room}) - ${nights}N</span>
    //                     <span>₹${total.toLocaleString('en-IN')}</span>
    //                 </li>`;
    //         }

    //         subtotal += total;
    //         nightsTotal += nights;
    //     });

    //     const total = subtotal;

    //     // Calculate end date
    //     const startDate = $('#start-date').val();
    //     if (startDate && nightsTotal > 0) {
    //         const d = new Date(startDate);
    //         d.setDate(d.getDate() + nightsTotal);
    //         $('#custom-package-form').data('end-date', d.toISOString().split('T')[0]);
    //     }

    //     $('#summary-itinerary').html(itineraryHtml ? `<ul class="space-y-1">${itineraryHtml}</ul>` : '<p class="text-sm text-slate-500">Add items...</p>');
    //     $('#summary-subtotal').text(`₹${subtotal.toLocaleString('en-IN')}`);
    //     $('#summary-total').text(`₹${total.toLocaleString('en-IN')}`);
    // }

    function updateSummary(){
        let subtotal = 0;
        let validRooms = 0;
        let itineraryHtml = '';

        $('#itinerary-rows .itinerary-row').each(function(){
            const hotel = $(this).find('.hotel-select option:selected').text();
            const room = $(this).find('.room-select option:selected').text();
            const nights = parseInt($(this).find('.nights-input').val() || 1);
            const price = parseFloat($(this).find('.room-select option:selected').attr('data-price') || 0);

            if (price > 0) {
                validRooms++;
                subtotal += price * nights;

                // show rows only if >= 2 rooms
                itineraryHtml += `<li class="flex justify-between text-sm">
                    <span>${hotel} (${room}) - ${nights}N</span>
                    <span>₹${(price * nights).toLocaleString('en-IN')}</span>
                </li>`;
            }
        });

        // ❌ less than 2 rooms → hide pricing
        if (validRooms < 2) {
            $('#summary-itinerary').html('<p class="text-sm text-slate-500">Select minimum 2 rooms to see total</p>');
            $('#summary-subtotal').text('₹0');
            $('#summary-total').text('₹0');
            return;
        }

        // ✅ show combined total
        // $('#summary-itinerary').html(`<ul class="space-y-1">${itineraryHtml}</ul>`);
        $('#summary-subtotal').text(`₹${subtotal.toLocaleString('en-IN')}`);
        $('#summary-total').text(`₹${subtotal.toLocaleString('en-IN')}`);
    }


    // -------------------------------
    // SUBMIT INQUIRY
    // -------------------------------
    $('#submit-inquiry').on('click', function(e){
        e.preventDefault();

        const pkgName = $('#package-name').val().trim();
        const destinationId = $('#destination').val();
        const startDate = $('#start-date').val();
        const endDate = $('#custom-package-form').data('end-date') || null;
        const adults = $('#adults').val();
        const children = $('#children').val();
        const subtotal = $('#summary-subtotal').text().replace('₹', '').trim();
        const total = $('#summary-total').text().replace('₹', '').trim();

        // ✅ Step 1: Basic validation before AJAX
        if (pkgName === '') {
            Swal.fire('Missing!', 'Please enter package name.', 'warning');
            return;
        }
        if (!destinationId || destinationId == 0) {
            Swal.fire('Missing!', 'Please select a destination.', 'warning');
            return;
        }
        if (!startDate) {
            Swal.fire('Missing!', 'Please select start date.', 'warning');
            return;
        }

        // Collect itinerary rows
        let items = [];
        $('#itinerary-rows .itinerary-row').each(function(){
            let cityId = $(this).find('.city-select').val();
            let cityName = $(this).find('.city-select option:selected').text();
            let hotelId = $(this).find('.hotel-select').val();
            let hotelName = $(this).find('.hotel-select option:selected').text();
            let roomId = $(this).find('.room-select').val();
            let roomName = $(this).find('.room-select option:selected').text();
            let nights = $(this).find('.nights-input').val();
            let price = $(this).find('.room-select option:selected').data('price') || 0;
            let total_price = price * nights;

            if (cityId > 0 && hotelId > 0 && roomId > 0) {
                items.push({
                    city_id: cityId,
                    city_name: cityName,
                    hotel_id: hotelId,
                    hotel_name: hotelName,
                    room_id: roomId,
                    room_name: roomName,
                    room_price: price,
                    nights: nights,
                    total_price: total_price
                });
            }
        });

        // ✅ Step 2: Validate at least one itinerary row
        if (items.length === 0) {
            Swal.fire('Add Itinerary!', 'Please add at least one valid itinerary row.', 'info');
            return;
        }

        // ✅ Step 3: Disable button during processing
        const $btn = $(this);
        $btn.prop('disabled', true).html('<i class="bi bi-hourglass-split"></i> Saving...');

        // ✅ Step 4: AJAX Save
        $.ajax({
            url: baseUrl + 'website/save_custom_inquiry',
            type: 'POST',
            dataType: 'json',
            data: {
                package_name: pkgName,
                destination_id: destinationId,
                start_date: startDate,
                end_date: endDate,
                no_of_adults: adults,
                no_of_children: children,
                subtotal: subtotal,
                total: total,
                items: JSON.stringify(items)
            },
            success: function(res) {
                if (res.status === 'success') {
                        Swal.fire({
                            title: 'Inquiry Saved!',
                            text: 'Your custom package inquiry was submitted successfully.',
                            icon: 'success',
                            confirmButtonText: 'View PDF'
                        }).then(() => {
                            // ✅ Redirect to PDF page
                            window.location.href = baseUrl + 'inquiry-pdf/' + res.inquiry_id;
                        });
                    } else {
                        Swal.fire('Error', 'Something went wrong while saving.', 'error');
                    }
                },

            error: function(){
                Swal.fire('Error', 'Unable to save inquiry. Please try again.', 'error');
            },
            complete: function(){
                $btn.prop('disabled', false).html('Submit Inquiry');
            }
        });
    });

});

function fetchRoomPrice(row){
    const roomId = row.find('.room-select').val();
    const startDate = $('#start-date').val();

    if (!startDate || !roomId || roomId == 0) return;

    $.ajax({
        url: baseUrl + 'website/get_room_price',
        type: 'POST',
        data: { room_id: roomId, start_date: startDate },
        dataType: 'json',
        success: function(res){
            const price = parseFloat(res.price_with_markup || 0);
            row.find('.room-select option:selected').attr('data-price', price);
            updateSummary();
        }
    });
}

</script>
